<?php
/**
 * Function describe for ZenOffice 
 * 
 * @package zenoffice
 */

/**
 * Set a constant that holds the theme's minimum supported PHP version.
 */
define( 'ZENOFFICE_MIN_PHP_VERSION', '7.0' );

/**
 * Immediately after theme switch is fired we we want to check php version and
 * revert to previously active theme if version is below our minimum.
 */
add_action( 'after_switch_theme', 'zenoffice_test_for_min_php' );

/**
 * Switches back to the previous theme if the minimum PHP version is not met.
 */
function zenoffice_test_for_min_php() {

	// Compare versions.
	if ( version_compare( PHP_VERSION, ZENOFFICE_MIN_PHP_VERSION, '<' ) ) {
		// Site doesn't meet themes min php requirements, add notice...
		add_action( 'admin_notices', 'zenoffice_min_php_not_met_notice' );
		// ... and switch back to previous theme.
		switch_theme( get_option( 'theme_switched' ) );
		return false;

	};
}

if ( ! function_exists( 'wp_body_open' ) ) {
        function wp_body_open() {
                do_action( 'wp_body_open' );
        }
}

/**
 * An error notice that can be displayed if the Minimum PHP version is not met.
 */
function zenoffice_min_php_not_met_notice() {
	?>
	<div class="notice notice-error is_dismissable">
		<p>
			<?php esc_html_e( 'You need to update your PHP version to run this theme.', 'zenoffice' ); ?> <br />
			<?php
			printf(
				/* translators: 1 is the current PHP version string, 2 is the minmum supported php version string of the theme */
				esc_html__( 'Actual version is: %1$s, required version is: %2$s.', 'zenoffice' ),
				PHP_VERSION,
				ZENOFFICE_MIN_PHP_VERSION
			); // phpcs: XSS ok.
?>
		</p>
	</div>
	<?php
}

/**
 * Fix skip link focus in IE11.
 *
 * This does not enqueue the script because it is tiny and because it is only for IE11,
 * thus it does not warrant having an entire dedicated blocking script being loaded.
 *
 * @link https://git.io/vWdr2
 */
function zenoffice_skip_link_focus_fix() {
	// The following is minified via `terser --compress --mangle -- js/skip-link-focus-fix.js`.
	?>
	<script>
	/(trident|msie)/i.test(navigator.userAgent)&&document.getElementById&&window.addEventListener&&window.addEventListener("hashchange",function(){var t,e=location.hash.substring(1);/^[A-z0-9_-]+$/.test(e)&&(t=document.getElementById(e))&&(/^(?:a|select|input|button|textarea)$/i.test(t.tagName)||(t.tabIndex=-1),t.focus())},!1);
	</script>
	<?php
}
add_action( 'wp_print_footer_scripts', 'zenoffice_skip_link_focus_fix' );

if ( ! function_exists( 'zenoffice_setup' ) ) :
	/**
	 * ZenOffice setup.
	 */
	function zenoffice_setup() {

	    // add WooCommerce support
		add_theme_support( 'woocommerce' );
	}
endif; // zenoffice_setup
add_action( 'after_setup_theme', 'zenoffice_setup' );



if ( ! function_exists( 'zenoffice_sanitize_url' ) ) :

	function zenoffice_sanitize_url( $url ) {
		return esc_url_raw( $url );
	}

endif; // zenoffice_sanitize_url

if ( ! function_exists( 'zenoffice_load_css_and_scripts' ) ) :

	function zenoffice_load_css_and_scripts() {

		wp_enqueue_style( 'zenoffice-stylesheet', get_template_directory_uri() . '/style.css' );
		wp_enqueue_style( 'zenoffice-child-style', get_stylesheet_uri(), array( 'zenoffice-stylesheet' ) );

		

		wp_enqueue_script( 'classie-js',
			get_stylesheet_directory_uri() . '/assets/js/classie.js',
			array() );

		wp_enqueue_script( 'uisearch-js',
			get_stylesheet_directory_uri() . '/assets/js/uisearch.js',
			array( 'jquery', 'classie-js' ) );

		wp_enqueue_script( 'zenoffice-utilities-js',
			get_stylesheet_directory_uri() . '/assets/js/utilities.js',
			array( 'jquery', 'uisearch-js' ) );
	}

endif; // zenoffice_load_css_and_scripts

add_action( 'wp_enqueue_scripts', 'zenoffice_load_css_and_scripts' );

if ( ! class_exists( 'zenoffice_Customize' ) ) :
	/**
	 * Singleton class for handling the theme's customizer integration.
	 */
	final class zenoffice_Customize {

		// Returns the instance.
		public static function get_instance() {

			static $instance = null;

			if ( is_null( $instance ) ) {
				$instance = new self;
				$instance->setup_actions();
			}

			return $instance;
		}

		// Constructor method.
		private function __construct() {}

		// Sets up initial actions.
		private function setup_actions() {

			// Register panels, sections, settings, controls, and partials.
			add_action( 'customize_register', array( $this, 'sections' ) );

			// Register scripts and styles for the controls.
			add_action( 'customize_controls_enqueue_scripts', array( $this, 'enqueue_control_scripts' ), 0 );
		}

		// Sets up the customizer sections.
		public function sections( $manager ) {

			// Load custom sections.

			// Register custom section types.
			$manager->register_section_type( 'zenlife_Customize_Section_Pro' );

			// Register sections.
			$manager->add_section(
				new zenlife_Customize_Section_Pro(
					$manager,
					'zenoffice',
					array(
						'title'    => esc_html__( 'ZenOfficePro', 'zenoffice' ),
						'pro_text' => esc_html__( 'Upgrade', 'zenoffice' ),
						'pro_url'  => esc_url( 'https://zentemplates.com/product/zenofficepro' )
					)
				)
			);
		}

		// Loads theme customizer CSS.
		public function enqueue_control_scripts() {

			wp_enqueue_script( 'zenlife-customize-controls', trailingslashit( get_template_directory_uri() ) . 'js/customize-controls.js', array( 'customize-controls' ) );

			wp_enqueue_style( 'zenlife-customize-controls', trailingslashit( get_template_directory_uri() ) . 'css/customize-controls.css' );
		}
	}
endif; // zenoffice_Customize

// Doing this customizer thang!
zenoffice_Customize::get_instance();

/**
 * Remove Parent theme Customize Up-Selling Section
 */
if ( ! function_exists( 'zenoffice_remove_parent_theme_upsell_section' ) ) :

	function zenoffice_remove_parent_theme_upsell_section( $wp_customize ) {

		// Remove Parent-Theme Upsell section
		$wp_customize->remove_section('zenlife');
	}

endif; // zenoffice_remove_parent_theme_upsell_section

add_action( 'customize_register', 'zenoffice_remove_parent_theme_upsell_section', 100 );

if ( ! function_exists( 'zenoffice_show_social_sites' ) ) :

	function zenoffice_show_social_sites() {

		$socialURL = get_theme_mod('zenoffice_social_facebook');
		if ( !empty($socialURL) ) {

			echo '<li><a href="' . esc_url( $socialURL ) . '" title="' . esc_attr( __('Follow us on Facebook', 'zenoffice') ) . '" class="facebook16"></a></li>';
		}

		$socialURL = get_theme_mod('zenoffice_social_twitter');
		if ( !empty($socialURL) ) {

			echo '<li><a href="' . esc_url( $socialURL ) . '" title="' . esc_attr( __('Follow us on Twitter', 'zenoffice') ) . '" class="twitter16"></a></li>';
		}

		$socialURL = get_theme_mod('zenoffice_social_instagram');
		if ( !empty($socialURL) ) {

			echo '<li><a href="' . esc_url( $socialURL ) . '" title="' . esc_attr( __('Follow us on Instagram', 'zenoffice') ) . '" class="instagram16"></a></li>';
		}

		$socialURL = get_theme_mod('zenoffice_social_rss');
		if ( !empty($socialURL) ) {

			echo '<li><a href="' . esc_url( $socialURL ) . '" title="' . esc_attr( __('Follow our RSS Feeds', 'zenoffice') ) . '" class="rss16"></a></li>';
		}
	}
endif; // zenoffice_show_social_sites

if ( ! function_exists( 'zenoffice_customize_register' ) ) :

	/**
	 * Register theme settings in the customizer
	 */
	function zenoffice_customize_register( $wp_customize ) {

		/**
		 * Add Social Sites Section
		 */
		$wp_customize->add_section(
			'zenoffice_social_section',
			array(
				'title'       => __( 'Social Sites', 'zenoffice' ),
				'capability'  => 'edit_theme_options',
			)
		);
		
		// Add facebook url
		$wp_customize->add_setting(
			'zenoffice_social_facebook',
			array(
			    'sanitize_callback' => 'zenoffice_sanitize_url',
			)
		);

		$wp_customize->add_control( new WP_Customize_Control( $wp_customize, 'zenoffice_social_facebook',
	        array(
	            'label'          => __( 'Facebook Page URL', 'zenoffice' ),
	            'section'        => 'zenoffice_social_section',
	            'settings'       => 'zenoffice_social_facebook',
	            'type'           => 'text',
	            )
	        )
		);

		// Add Twitter url
		$wp_customize->add_setting(
			'zenoffice_social_twitter',
			array(
			    'sanitize_callback' => 'zenoffice_sanitize_url',
			)
		);

		$wp_customize->add_control( new WP_Customize_Control( $wp_customize, 'zenoffice_social_twitter',
	        array(
	            'label'          => __( 'Twitter URL', 'zenoffice' ),
	            'section'        => 'zenoffice_social_section',
	            'settings'       => 'zenoffice_social_twitter',
	            'type'           => 'text',
	            )
	        )
		);

		// Add Instagram url
		$wp_customize->add_setting(
			'zenoffice_social_instagram',
			array(
			    'sanitize_callback' => 'zenoffice_sanitize_url',
			)
		);

		$wp_customize->add_control( new WP_Customize_Control( $wp_customize, 'zenoffice_social_instagram',
	        array(
	            'label'          => __( 'LinkedIn URL', 'zenoffice' ),
	            'section'        => 'zenoffice_social_section',
	            'settings'       => 'zenoffice_social_instagram',
	            'type'           => 'text',
	            )
	        )
		);

		// Add RSS Feeds url
		$wp_customize->add_setting(
			'zenoffice_social_rss',
			array(
			    'sanitize_callback' => 'zenoffice_sanitize_url',
			)
		);

		$wp_customize->add_control( new WP_Customize_Control( $wp_customize, 'zenoffice_social_rss',
	        array(
	            'label'          => __( 'RSS Feeds URL', 'zenoffice' ),
	            'section'        => 'zenoffice_social_section',
	            'settings'       => 'zenoffice_social_rss',
	            'type'           => 'text',
	            )
	        )
		);
	}
endif; // zenoffice_customize_register

add_action('customize_register', 'zenoffice_customize_register');