<?php

/**
 * Soundscape functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Soundscape
 * @since Soundscape 1.0
 */


if (! function_exists('soundscape_support')) :

	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	function soundscape_support()
	{

		// Add support for block styles.
		add_theme_support('wp-block-styles');

		// Enqueue editor styles.
		add_editor_style('style.css');

		// Remove patterns that ship with WordPress Core.
		remove_theme_support('core-block-patterns');
	}

endif;

add_action('after_setup_theme', 'soundscape_support');


/**
 * Enqueue block stylesheets.
 */

if (! function_exists('soundscape_block_stylesheets')) :
	/**
	 * Enqueue custom block stylesheets
	 *
	 * @since Soundscape 1.0
	 * @return void
	 */
	function soundscape_block_stylesheets()
	{
		/**
		 * The wp_enqueue_block_style() function allows us to enqueue a stylesheet
		 * for a specific block. These will only get loaded when the block is rendered
		 * (both in the editor and on the front end), improving performance
		 * and reducing the amount of data requested by visitors.
		 *
		 * See https://make.wordpress.org/core/2021/12/15/using-multiple-stylesheets-per-block/ for more info.
		 */
		wp_enqueue_block_style(
			'core/button',
			array(
				'handle' => 'soundscape-button-style-outline',
				'src'    => get_parent_theme_file_uri('assets/css/button-outline.css'),
				'ver'    => wp_get_theme(get_template())->get('Version'),
				'path'   => get_parent_theme_file_path('assets/css/button-outline.css'),
			)
		);
	}
endif;

add_action('init', 'soundscape_block_stylesheets');

/**
 * Register pattern categories.
 */

if (! function_exists('soundscape_pattern_categories')) :
	/**
	 * Register pattern categories
	 *
	 * @since Soundscape 1.0
	 * @return void
	 */
	function soundscape_pattern_categories()
	{

		register_block_pattern_category(
			'soundscape_page',
			array(
				'label'       => _x('Pages', 'Block pattern category', 'soundscape'),
				'description' => __('A collection of full page layouts.', 'soundscape'),
			)
		);
	}
endif;

add_action('init', 'soundscape_pattern_categories');


if (! function_exists('soundscape_styles')) :

	/**
	 * Enqueue styles.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */

	function soundscape_styles()
	{
		// Enqueue theme stylesheet.
		wp_enqueue_style(
			'soundscape-style',
			get_template_directory_uri() . '/style.css',
			array(),
			filemtime(get_theme_file_path('style.css'))
		);

		wp_enqueue_script(
			'soundscape-script',
			get_theme_file_uri('assets/js/custom.js'),
			array(),
			filemtime(get_theme_file_path('assets/js/custom.js')),
			true
		);
	}

endif;

add_action('wp_enqueue_scripts', 'soundscape_styles');

if (! function_exists('soundscape_block_editor_styles')) :

	/**
	 * Enqueue rtl editor styles
	 */

	function soundscape_block_editor_styles()
	{
		if (is_rtl()) {
			wp_enqueue_style(
				'soundscape-rtl',
				get_theme_file_uri('rtl.css'),
				array(),
				filemtime(get_theme_file_path('rtl.css'))
			);
		}
	}

endif;

add_action('enqueue_block_editor_assets', 'soundscape_block_editor_styles');

// Add block patterns
require get_template_directory() . '/inc/block-patterns.php';

// Add block styles
require get_template_directory() . '/inc/block-styles.php';

// Block Filters
require get_template_directory() . '/inc/block-filters.php';

// Svg icons
require get_template_directory() . '/inc/icon-function.php';

// Theme About Page
require get_template_directory() . '/inc/about.php';


